% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program CalibrateGamma.m
function CalibrateGamma (rgb)

%% Display Setup Module
% Define display parameters
whichScreen = max(Screen('screens')); 
p.ScreenGamma = 1;	 % set it to 1 so we can measure real gamma
p.ScreenBackground = 0;
if nargin < 1 || isempty(rgb); rgb = [1 1 1]; end

% Open display window and hide the mouse cursor
if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
PsychImaging('PrepareConfiguration');
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange'); 
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % open a display window
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);
Screen('BlendFunction', windowPtr, 'GL_ONE', 'GL_ZERO', [rgb 1]);
HideCursor;

% Set screen font
fontsz = round(36 / 1024 * p.ScreenRect(4) / 2) * 2;
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, fontsz);
% Control keys
keys={'left' 'right' 'up' 'down' 'return' 'enter' 'esc'};
% Instruction 1
str = [ 'View the display from a distance so that you do not' ...
       ' see any stripes.\n\n Adjust the luminance of the ' ...
       ' middle part until it matches that of the flankers.' ...
       ' \n\n' 'Press an arrow key to continue.'];
DrawFormattedText(windowPtr, str, 100, 200, 1, 48, 0, 0, 1.5);
Screen('Flip', windowPtr);
WaitTill(keys(1 : 4)); % wait for a key to start
 
% Instruction 2
str = ['Use Left/right arrow keys for coarse adjustments,\n' ...
       'and Up/down keys for fine adjustments.\n' ...
       'Press Enter key when you are done'];
DrawFormattedText(windowPtr, str, 100, 200, 255, [], 0, 0, ...
       1.5);
Screen('Flip', windowPtr, 0, 1);
 
%% Experimental Module
% Specify the stimulus
sz = 128; % size of the middle patch in pixels
 
% Compute and set up the test stimulus
lum = linspace(0, 1, 17);
vol = lum .^ (1 / 2.2);     % initial guess of Gamma is 2.2
hI = [17 17 9 17 13 9 5 17 15 13 11 9 7 5 3]; 
                            % high value index
lI = [1 9  1 13 9 5  1 15 13 11 9 7 5 3 1]; % low value index
step = [0.3 3] / 255;       % fine and coarse steps
img = zeros(sz, sz * 3);
rect = CenterRect([0 0 1 1] * sz, p.ScreenRect);
tex = zeros(1, 2);
 
% Loop through 15 steps in a fixed order
for i = 1 : 15
    high = vol(hI(i)); 
    low = vol(lI(i));
    ind = (hI(i) + lI(i)) / 2;
    img(:) = high; img(2 : 2 : sz, :) = low;
    tex(1) = Screen('MakeTexture', windowPtr, img, 0, 0, 2);
    tex(2) = Screen('MakeTexture', windowPtr, ...
              img([2 : sz 1], :), 0, 0, 2);
    mid = vol(ind);         % start with guessed value
    
    while 1
        KbReleaseWait;      % avoid continuous change
        while 1
            for j = 1 : 2   % generate flickering flankers
                Screen('DrawTexture', windowPtr, tex(j));
                Screen('FillRect', windowPtr, mid, rect);
                Screen('Flip', windowPtr, 0, 1);
            end
            key = ReadKey(keys);
            if ~isempty(key), break; end
        end
        switch key
            case 'left',    mid = mid - step(2);
            case 'right',   mid = mid + step(2);
            case 'up',      mid = mid + step(1);
            case 'down',    mid = mid - step(1);
            case 'esc',     sca; error('ESC pressed.');
            otherwise % return or enter
                if i == 1
                    guessGamma = log(0.5) / log(mid);
                    vol = lum .^ (1 / guessGamma);
                else
                    vol(ind) = mid;
                end
                Beeper;
                break; % go to next step
        end
        
        if mid > 1 || mid < 0
            Beeper; % out of range warning
            mid=max(0, min(mid, 1));
        end
    end 
end

% Fit Gamma
costfunc = @(x) sum((lum .^ (1 / x) - vol) .^ 2);
gamma = fminsearch(costfunc, guessGamma);
rsq = 1 - costfunc(gamma) / var(vol, 1) / 17;
% plot the result
figure(3);
x = linspace(0, 1, 256); 
plot(x, x .^ gamma, 'r'); hold on;
plot(vol, lum, 'o'); hold off; 
xlabel('Normalized Output'); 
ylabel('Normalized Luminance');
text(0.2, 0.9, sprintf('RGB = [%g %g %g]', rgb));
text(0.2, 0.8, sprintf('Gamma = %.3g', gamma));
text(0.2, 0.7, sprintf('Rsq = %.4g', rsq));



